# AI Coding Instructions for Soal Online (Online Exam System)

## Architecture Overview

This is a **PHP-based online exam/quiz system** for Indonesian students (Matematika/Mathematics focus). The application supports multiple question types and role-based access (students & admins).

### Core Components

- **Question Types**: 4 formats managed in separate tables and display modules
  - `ganda_b` - Multiple choice (single answer)
  - `komplek` - Complex multiple choice (multiple answers)
  - `jodoh` - Matching questions (kolom_kiri/kolom_kanan columns)
  - `bs` - True/False questions

- **Result Storage**: Parallel tables track answers and scores
  - `hasil_ganda_b`, `hasil_komplek`, `hasil_jodoh`, `hasil_bs`
  - Each stores: `username`, `id_soal`, `jawaban_user`, `skor`, timestamps

- **User Management**:
  - Students: `users` table (username/password_hashed)
  - Admin: Session-based with hardcoded credentials in `admin.php` (username: "admin", password: "admin123")
  - Token system: `token_soal` table controls exam access (active tokens only)

### Data Flow

```
Login (login.php) 
  → Token validation (cek_token.php)
  → Exam start (ganda_B.php)
  → Answer submission (update hasil_* tables)
  → Score calculation (skor_akhir.php)
  → Reports (ambil_*.php)
```

## Critical Developer Workflows

### Database Credentials (Embedded)
**Location**: `koneksi.php`, `login.php`, multiple answer handlers
```php
$host = "localhost";
$user = "lnibbicy_kka4";
$pass = "magetan#2";
$db = "lnibbicy_kka4";
```
**Pattern**: Direct MySQLi connection (NOT using abstraction). DO NOT commit credentials; extract to environment.

### Answer Submission & Scoring
- **Single submission**: Files like `input_ganda_b.php` save answers to result tables
- **Duplicate handling**: `skor_akhir.php` groups by `id_soal` to prevent double-counting retakes
- **Scoring logic**: Usually binary (correct/incorrect) or point-based depending on question type
- **Key files**: 
  - `simpan_jawaban.php` - Answer persistence
  - `submit_jawaban.php` - Form submission handler
  - `koreksi_jawaban.php` - Answer verification

### Admin Workflows
- Token management: `admin_token.php` (generate/activate/deactivate exam tokens)
- Question input: `input_ganda_b.php`, `input_komplek.php`, `input_jodoh.php`, `input_bs.php`
- Monitoring: `monitoring.php` (live exam status), `admin_status.php` (exam progress)
- Reports: `rekap_akm.php` (aggregate results), individual reports via `ambil_*.php`
- User management: `insert_user.php` (registration), `verifikasi_user.php` (approval), `daftar_peserta.php` (removal)

## Code Patterns & Conventions

### Session Security
```php
session_start();
if (!isset($_SESSION["user"])) {
    header("Location: login.php");
    exit();
}
$user = $_SESSION["user"];
```
**Pattern**: Every protected page starts with session check redirecting to login.php. Use `$_SESSION["level"] === "admin"` for role validation.

### MySQLi with Prepared Statements
**Good** (used in login.php):
```php
$stmt = $mysqli->prepare("SELECT password FROM users WHERE username = ?");
$stmt->bind_param("s", $username);
```
**Avoid**: String interpolation in queries (seen in older code like skor_akhir.php).

### Question/Answer Display Pattern
Questions are fetched by type, then displayed in a loop. Example from `ganda_B.php`:
- Get unanswered question: `WHERE id NOT IN (SELECT id_soal FROM hasil_ganda_b WHERE username = ?)`
- Display options (typically a, b, c, d columns in table)
- Store answer in `hasil_ganda_b` with timestamp

### Token System
- Tokens must be `aktif=1` in `token_soal` table
- `admin_token.php` generates 6-char alphanumeric tokens (no '0')
- Login requires token submission before password verification
- Tokens auto-expire based on `waktu_expired` column

### Report Generation
- PDF export via DomPDF library (see `generate_pdf.php`, `cetak_template.php`)
- Excel export via `export_xls.php`
- Reports query result tables with `SELECT DISTINCT username` to get participant lists

## Project-Specific Conventions

### File Organization
- **Root**: Main pages (login, exam flow, score display)
- **admin.php**: Admin menu hub
- **input_*.php**: Question entry forms (admin only)
- **ambil_*.php**: Report generation (one per question type)
- **soal_online/**: Alternative/legacy implementation

### Naming Conventions
- Question type abbreviations: `ganda_b`, `komplek`, `jodoh`, `bs`
- Answer storage: `hasil_<question_type>` tables
- Admin flows: `admin_*.php`
- Report/retrieval: `ambil_*.php`

### HTML Structure
- Bootstrap 5.3.7 CSS included (multiple bootstrap-*.css files)
- Inline styles preferred over external CSS in content pages
- RTL (right-to-left) CSS variants for Indonesian language support
- Tailwind CSS used in newer report pages (`ambil_*.php`)

## External Dependencies

- **DomPDF** (v0.6.2) - PDF generation via Composer
- **Barcode Generator** (v3.2) - Question/exam identification
- **Bootstrap 5.3.7** - UI framework
- **Tailwind CSS** - Alternative styling in reports
- **MySQL** - Database (PHP 8.0.30+ platform)

## Integration Points

### Database Tables
- `users` - Student credentials
- `ganda_b`, `komplek`, `jodoh`, `bs` - Question content
- `hasil_ganda_b`, `hasil_komplek`, `hasil_jodoh`, `hasil_bs` - Student answers & scores
- `token_soal` - Exam session tokens

### Cross-File Communication
- Session variables: `$_SESSION["user"]`, `$_SESSION["level"]` (admin check)
- Query parameters: `?question=<id>` for direct question navigation
- POST data: `$_POST['jawaban']` for answer submission, `$_POST['token']` for token entry

## Important Gotchas & Gotchas

1. **Admin hardcoded credentials**: `admin.php` uses plaintext comparison, NOT hashed passwords
2. **Multiple question type systems**: Changes to question structure require updates to 4 separate tables AND their input forms AND result retrieval
3. **No ORM/migrations**: Schema changes must be manual SQL updates
4. **Credentials hardcoded**: Extract to environment before production deployment
5. **Legacy code coexistence**: Both procedural PHP and some DomPDF-based approaches mixed
6. **Duplicate answer handling**: Always use `GROUP BY id_soal` when aggregating user results to avoid counting retakes multiple times

## Before Making Changes

- Identify which question type(s) are affected (`ganda_b`, `komplek`, `jodoh`, or `bs`)
- Check both the input form (`input_*.php`), result table (`hasil_*.php`), and report retrieval (`ambil_*.php`)
- Test with actual database since no ORM abstracts schema
- Verify session/auth before deployment (especially admin pages)
- Ensure prepared statements used for any new database queries
